/*
 *  LICENSE:
 *  Copyright (c) 2008 Freescale Semiconductor
 *  
 *  Permission is hereby granted, free of charge, to any person 
 *  obtaining a copy of this software and associated documentation 
 *  files (the "Software"), to deal in the Software without 
 *  restriction, including without limitation the rights to use, 
 *  copy, modify, merge, publish, distribute, sublicense, and/or 
 *  sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following 
 *  conditions:
 *  
 *  The above copyright notice and this permission notice 
 *  shall be included in all copies or substantial portions 
 *  of the Software.
 *  
 *  THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, 
 *  EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES 
 *  OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND 
 *  NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT 
 *  HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, 
 *  WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, 
 *  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 *  
 *
 *  File: $Id: deb_tools.h,v 1.4 2008/08/01 20:57:20 b17778 Exp $
 *
 */


/////////////////////////////////////////////////////////////////////////////////////////////
//                                                                                         //
// This AXE task is a sample code for a FIR filter. It compiles for both:                  //    
//    1. a stand alone task for the AXE                                                    //
//    2. an AXE scheduler task                                                             //
//                                                                                         //
// As a stand alone task, there happens nothing more than taking a set of input data       //
// and filter it with a simple FIR algorithm.                                              //
//                                                                                         //
// As an AXE scheduler task it is a little more complex, though we have to implement the    //
// interprocess communication between the AXE task and calling PPC application. 	   //
// Furthermore,   //
// the AXE scheduler provides special function for allocating memory and      //
// console outputs.  //
// However, here is what this AXE scheduler task does:                                     //
//                                                                                         //
//    1.  Directly from the beginning an endless loop is executed, that waits for           //
//        messages. This is not(!) busy waiting, due to the fact that the AXE scheduler    //
//        puts a task, which waits for a message in an inactive state and awakes it        //
//        when a message for that task arrives.                                            //
//    2.  When a message arrives, the task switches to a routine specified by the message      //
//        type. In the beginning we should receive an initialization message.              //
//    3.  The initialization message contains pointers to an input and an output buffer.   //
//        Furthermore a pointer to a field of coefficients needed by the FIR algorithm.    //
//    4.  The initialization process sets the global variables of this AXE scheduler task  //
//        to the corresponding buffers.                                                    //
//    5.  After initialization the task waits for a message that starts the calculation.   //
//        It is assumed that the calling PPC process fills the input buffer. On            //
//        reception of a message of type "calcualate" the task reads the input buffer,     //
//        filters the data, and writes the result to the output buffer.                    //
//    6.  Step 5 can be repeated now in an endlees loop until all data is processed.       //
//                                                                                         //
/////////////////////////////////////////////////////////////////////////////////////////////

//INCLUDES

/*
 * in taskcfg.h it is defined wether the task is a stand alone 
 * or a scheduler task
 */ 
#include "taskcfg.h"

/*
 * AXEInt32 has to be defined for the stand alone task
 */
#ifdef STAND_ALONE
typedef int AXEInt32;
#endif /* STAND_ALONE */


#ifdef AXE_SCHEDULER_TASK
/*
 * includes the AXE specific types
 */
#include "axetypes.h"

/*
 * includes the AXE scheduler API
 */
#include "axeapi.h"

#endif /* AXE_SCHEDULER_TASK */

/*
 * include the FIR specific types
 */
#include "fir.h"

#ifdef AXE_SCHEDULER_TASK

/*
 * the fir_ipc.h (FIR inter processor communication) header is the one file shared by
 * the ppc application and the AXE scheduler task. All the message types and 
 * formats are defined here.
 */
#include "fir_ipc.h"

#endif /* AXE_SCHEDULER_TASK */



#ifdef STAND_ALONE
#include "firdata.h"
#endif /* STAND_ALONE */

/*
 * common includes
 */
#include <stdio.h>
#include <string.h>



//FUNCTION DECLARATIONS


#ifdef AXE_SCHEDULER_TASK

/**
 *	\brief	 handles incoming messages
 *	\param[in] a pointer to the messages to be handeled
 *	\return    void
 *
 *  \note
 *
 *  This is quite a generic function and can be easily extended by adding new cases
 *  But as cases refelect diferent message types the fir_ipc.h header might be updated
 *
 */
void handleIncomingMessage(MsgType* psMsg);

/**
 *	\brief	 displays errors
 *	\param[in] a pointer to the error's first char
 *	\return    void
 *
 */
void handleError(char* err);

#endif /* AXE_SCHEDULER_TASK */


/**
 *	\brief	 takes input from input buffer, filters it and writes it to output buffer 
 *	\return    void
 */
void ifa_fir(__aacFirInput aacInput, __sFirParams *psFirParams
			, __sFirState *psFirState, __aacFirOutput aacOutput
			, int cycles) __attribute__((ifa));
			

/*******************************************************************************************
 *********************************                    **************************************
 *********************************   IMPLEMENTATION   **************************************
 *********************************                    **************************************
 *******************************************************************************************
 */
			
#ifdef AXE_SCHEDULER_TASK
void handleIncomingMessage(MsgType* psMsg)
{	
	switch(psMsg->type) // switch message type
	{
		case CALCMSG:
			ifa_fir( 	//call filter function 
				((sCalcMessage*) psMsg)->aacInput,
				((sCalcMessage*) psMsg)->psFirParams,
				((sCalcMessage*) psMsg)->psFirState,
				((sCalcMessage*) psMsg)->aacOutput,
				((sCalcMessage*) psMsg)->iBufferCount
				);
			break;
		default:
			handleError("No valid message type");
			 
	}
	
	/* send reply message to PPC application */ 
	MessageIsProcessed(psMsg);

}

void handleError(char* err)
{
	CPrint(err);
}
#endif /* AXE_SCHEDULER_TASK */


/* 
 *
 *  here a scheme of the filters functionality
 *
 *  input(i) --->[z^-1]--->[z^-2]---> ... --->[z^-NumberCoef]---
 *            |         |         |        |                    | 
 *           coef_0    coef_1    coef_2   coef_3               coef_n-1
 *            |         |         |        |                    |
 *            v         v         v        v                    v
 *            |-------> + ------> + --..-> + -----------------> + -> output(i)
 *
 *  The shift register (z) is implemented as a circular buffer 
 *
 */
 
void ifa_fir(__aacFirInput aacInput, __sFirParams *psFirParams
, __sFirState *psFirState, __aacFirOutput aacOutput
, int cycles)
{	

	//VARIABLES

	//creating some shorter handles 
	int 		*state_z	= &(psFirState->iCircBufOffset); // circular buffer offset for state array
	int 	  	stages		= psFirState->iFilterStages;	 // number of state stages
	__accum    	*aacZ		= psFirState->aacZ;				 // state array
	int   		iCoefCount	= psFirParams->iCoefCount;		 // number of coefficients
	__fixed 	*afiCoef 	= psFirParams->afiCoef;			 // coefficient array
	
	int   i = 0, ii = 0;	//loop counters..	
	
	
	//ALGORITHM
	
	
	// loop through the input buffer
	for (i = 0; i<cycles; i++){
	
		/* output_i = coef_0 * input_i */
		*(aacOutput + i) = (*(afiCoef)) * (*(aacInput+i)); 
		
		//summation loop for coefficients 1 to n-2
		for (ii = 1; ii < iCoefCount - 1; ii++)
		{
			/* output_i = output_i + coef_ii * the next state in the circ buffer */
			*(aacOutput + i) += *(afiCoef+ii) * *(aacZ + *state_z); 
			
			/*
			 * wrap to the stages pointer if necessary 
			 */
			if (++(*state_z) >= stages)
			{
				*state_z = 0;	
			}
	
		}
	/* 
	 * output_i = output_i * coef_n-1 * last state
	 * we do this outside of the loop to avoid to 
	 * decrease the state point at the end
	 */	
	*(aacOutput + i) +=*(afiCoef+ii)  * *(aacZ + *state_z); 
	
	/* 
	 * The last state became the first. 
	 * Here we store the last input 
	 */
	*(aacZ + *state_z) = *(aacInput+i);
	
	}
#ifdef STAND_ALONE	
	//print out results	
	for(i=0;i<cycles;i++)
	{
		printf("%a\n",*(aacOutput + i));
	}
#endif //STAND_ALONE

}

//MAIN FUNCTION
int main()
{
	#ifdef STAND_ALONE
	ifa_fir(myDynInput,&myDynParams,&myDynState
			, myDynOutput, BUFFER_COUNT);	
	while (1) {} // loop forever
	#endif //STAND_ALONE
	
	#ifdef AXE_SCHEDULER_TASK
		MsgType *psMsg;
		
		while (1) //waiting for messages
		{
			ReceiveMessage(&psMsg);
			handleIncomingMessage(psMsg);
		}	
	#endif /* AXE_SCHEDULER_TASK */
}




